
#include <hidef.h> /* for EnableInterrupts macro */
#include "derivative.h" /* include peripheral declarations */
#include "PublicTypes.h"


#include "GlucoseMeter.h"
#include "SwTimer.h"
#include "SystemConfig.h"
#include "SerialCommands.h"
#include "OPAMP.h"
#include "virtual_com.h"
#include "usb_cdc.h"
#include "MCU.h"        /* MCU Specific */




#if DEBUG_MODE==FALSE
//#define SEND_SINE_WAVE

#define CONNECTED_TO_USB	1		//PTADD_PTADD0 == 1	//Change this
#define CONNECTED_TO_LCD_BOARD	0	//IRQSC_IRQF == 1
#define GUI_TOLD_ME_TO_GO_STAND_ALONE	0//	PTADD_PTADD2	//change this

/* unsecure flash */
const uint_8 sec@0x040f = 0x00; 
/* checksum bypass */
const uint_8 checksum_bypass@0x040a = 0x0; 


//const unsigned char boot@0xFFBA = 0x00; // bypass checksum 

typedef union 
{
	byte Byte;
	struct 
	{
		byte ConnectedToUsb      	:1;
		byte ConnectedToLcdBoard	:1;
		byte ZigbeeEnabled			:1;
		byte BluetoothEnabled		:1;
		byte :1;
		byte :1;
		byte :1;
		byte :1;
	} Bits;
} SystemFlags;


typedef enum
{
	NO_MEASUREMENT,
	GLU_MEASUREMENT,
	BPM_MEASUREMENT,
	BPM_LEAK_TEST,
	ECG_MEASUREMENT,
	SPR_MEASUREMENT,
	HEIGHT_MEASUREMENT,
	WEIGHT_MEASUREMENT,
	TEMPERATURE_MEASUREMENT
};



UINT8 InBuffer[32];	//From the PC to the MCU
UINT8 OutBuffer[128];	//From the MCU to the PC
UINT8 OutSize;
UINT8 InSize;


/* Receive Buffer */
extern uint_8 g_curr_recv_buf[DATA_BUFF_SIZE];
/* Send Buffer */
extern uint_8 g_curr_send_buf[DATA_BUFF_SIZE];
/* Receive Data Size */
extern uint_8 g_recv_size;
/* Send Data Size */
extern uint_8 g_send_size;



UINT16 IdNumber = 0;
UINT8 MainActualState;
UINT8 ActualMeasurement;

UINT8 TimerSendDummyData;

void TimerSendDummyData_Event(void);
void EnableAFE(UINT8 measurementId);


typedef enum
{
	MAIN_STATE_DISCONNECTED,
	MAIN_STATE_CONTROLLED_BY_LCD_BOARD,
	MAIN_STATE_CONTROLLED_BY_USB
} MainStates_e;

/* States */
void StateDisconnected(void);
void StateControlledByLcdBoard(void);
void StateControlledByUsb(void);

/*****************************************************************************
 * Local Functions Prototypes
 *****************************************************************************/
static void Init_Sys(void);
static void Mcu_Init(void);
static void MCG_Init(void);


/* GlucoseMeter */
void GluStartMeasurementReq(void);
void GluAbortMeasurementReq(void);
void GluStartCalibrationReq(void);
void GlucometerBloodDetectedInd(void);
void GlucometerMeasurementOkInd(void);
void GlucometerCalibrationOkInd(void);
void GlucometerDebugNewDataReadyInd(void);


/* system restart */
void SystemRestartReq(void);


/* Serial comm */
void SerialComm_PeriodicTask(void);
void SerialComm_SendData(void);



/** Pointer to function that will be used as a callback */

const pFunc_t GlucoseMeter_Events[] = 
{
	NULL,						//EVENT_GLU_NONE,
	GlucometerBloodDetectedInd,	//EVENT_GLU_BLOOD_DETECTED,
	GlucometerMeasurementOkInd,	//EVENT_GLU_FINISH_MEASUREMENT_OK,
	GlucometerCalibrationOkInd,	//EVENT_GLU_FINISH_CALIBRATION_OK
	GlucometerDebugNewDataReadyInd,	//EVENT_GLU_DEBUG_NEW_DATA_READY
};


const pFunc_t MainStateMachine[] = 
{
	StateDisconnected,
	StateControlledByLcdBoard,
	StateControlledByUsb
};



const pFunc_t ExecuteCommandReq[] =
{
	GluStartMeasurementReq,				//GLU_START_MEASUREMENT,
	GluAbortMeasurementReq,				//GLU_ABORT_MEASUREMENT,
	GluStartCalibrationReq,				//GLU_START_CALIBRATION,
	NULL,								//GLU_BLOOD_DETECTED,
	NULL,								//GLU_MEASUREMENT_COMPLETE_OK,
	NULL,								//GLU_CALIBRATION_COMPLETE_OK,
	
	NULL,				//BPM_START_MEASUREMENT,
	NULL,				//BPM_ABORT_MEASUREMENT,
	NULL,								//BPM_MEASUREMENT_COMPLETE_OK,
	NULL,								//BPM_MEASUREMENT_ERROR,
										  
	NULL,				//BPM_START_LEAK_TEST,
	NULL,					//BPM_ABORT_LEAK_TEST,
	NULL,								//BPM_LEAK_TEST_COMPLETE,	
	
	NULL,	//ECG_HEART_RATE_START_MEASUREMENT,
	NULL,	//ECG_HEART_RATE_ABORT_MEASUREMENT,
	NULL,								//ECG_HEART_RATE_MEASUREMENT_COMPLETE_OK,
	NULL,								//ECG_HEART_RATE_MEASUREMENT_ERROR,
	NULL,								//ECG_HEART_BEAT_OCCURRED,
	
	NULL,//ECG_DIAGNOSTIC_MODE_START_MEASUREMENT,
	NULL, //ECG_DIAGNOSTIC_MODE_STOP_MEASUREMENT,
	NULL,								 //ECG_DIAGNOSTIC_MODE_NEW_DATA_READY,	
		
	NULL,				//TMP_READ_TEMPERATURE,
	NULL,					//HGT_READ_HEIGHT,
	NULL,					//WGT_READ_WEIGHT,

	NULL,	//SprStartMeasurementReq,				//SPR_START_MEASUREMENT,
	NULL,	//SprAbortMeasurementReq,				//SPR_ABORT_MEASUREMENT,
	NULL,								//SPR_MEASUREMENT_COMPLETE_OK,
	NULL,								//SPR_MEASUREMENT_ERROR,

	NULL,								//SPR_DIAGNOSTIC_MODE_START_MEASURMENT,
	NULL,								//SPR_DIAGNOSTIC_MODE_STOP_MEASURMENT,
	NULL,								//SPR_DIAGNOSTIC_MODE_NEW_DATA_READY,
	NULL,								//SPR_DIAGNOSTIC_MODE_MEASUREMENT_COMPLETE_OK,
	NULL,								//SPR_DIAGNOSTIC_MODE_MEASUREMENT_ERROR,
		
	NULL,				//POX_START_MEASUREMENT,
	NULL,				//POX_ABORT_MEASURMENT,
	NULL,								//POX_MEASURMENT_COMPLETE_OK,
	NULL,								//POX_MEASURMENT_ERROR,

	NULL,//POX_DIAGNOSTIC_MODE_START_MEASURMENT,
	NULL,//POX_DIAGNOSTIC_MODE_STOP_MEASURMENT,
	NULL,								//POX_DIAGNOSTIC_MODE_NEW_DATA_READY
	NULL,	//BPM_SEND_PRESSURE_VALUE_TO_PC,
	SystemRestartReq,	//SYSTEM_RESTART,
	NULL	//BPM_DATA_READY = 0xFF,
};



void main(void) 
{
    
  Init_Sys();
  AFE_ENABLE_PIN_INIT();
 		
  opamp_gp_mode();
	TRIAMP1C0_HighMode();
	TRIAMP2C0_HighMode(); 

	SwTimer_Init();      

	
	GlucoseMeter_Init();	
	EnableInterrupts;
  
 
  (void)TestApp_Init(); /* Initialize the USB Test Application */  
  TimerSendDummyData = SwTimer_CreateTimer(TimerSendDummyData_Event);
	
		
	for(;;) 
	{
		GlucoseMeter_PeriodicTask();
		
		
		SwTimer_PeriodicTask();
		SerialComm_PeriodicTask();
		MainStateMachine[MainActualState]();
	  __RESET_WATCHDOG(); /* feeds the dog */
	} /* loop forever */
}


static const UINT16 SinX[] = 
{
	1250, 1349, 1448, 1545, 1640, 1733, 1823, 1910, 1992, 2069, 2142, 2208, 2269, 2323, 2371, 2411, 2444, 2470, 2488, 2498, 2500, 2494, 2481, 2459, 2430, 2394, 2350, 2300, 2243, 2179, 2110, 2036, 1956, 1872, 1784, 1693, 1598, 1502, 1404, 1305, 1206, 1107, 1009, 912, 818, 726, 638, 553, 473, 398, 328, 264, 206, 155, 111, 73, 44, 21, 7, 0, 2, 11, 28, 53, 85, 124, 171, 225, 284, 350, 422, 499, 581, 667, 756, 849, 944, 1041, 1140, 1239
};

#define SIN_X_LAST_ELEMENT 	79
volatile UINT8 EcgBufferSize = 64;
volatile UINT8 Time	= 64;


void TimerSendDummyData_Event(void)
{
	if (ActualMeasurement == ECG_MEASUREMENT)
	{		
		static UINT8 SinXArrayActualElement = 0;

		//Send indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = ECG_DIAGNOSTIC_MODE_NEW_DATA_READY;
		OutBuffer[OutSize++] = EcgBufferSize + 2;				//data bytes
		
		OutBuffer[OutSize++] = (UINT8) (IdNumber >> 8);
		OutBuffer[OutSize++] = (UINT8) (IdNumber & 0x00FF);
		
		IdNumber++;
		
		
		while (OutSize < (EcgBufferSize + DATA_PACKET + 2))
		{
			OutBuffer[OutSize++] = SinX[SinXArrayActualElement] >> 8;		
			OutBuffer[OutSize++] = SinX[SinXArrayActualElement] & 0x00FF;
			
			if (SinXArrayActualElement == SIN_X_LAST_ELEMENT)
			{
				SinXArrayActualElement = 0;
			}
			else
			{
				SinXArrayActualElement++;
			}
		}

		(void)SerialComm_SendData();
		
		SwTimer_StartTimer(TimerSendDummyData, Time);
	}	
	
}

void StateDisconnected(void)
{
	if (CONNECTED_TO_USB)
	{
		MainActualState = MAIN_STATE_CONTROLLED_BY_USB;
	}
		else if (CONNECTED_TO_LCD_BOARD)
	{
		MainActualState = MAIN_STATE_CONTROLLED_BY_LCD_BOARD;
	}	
}


void StateControlledByLcdBoard(void)
{
	if (CONNECTED_TO_USB)
	{
		//Send LL16 to sleep
		//add code here
		MainActualState = MAIN_STATE_CONTROLLED_BY_USB;
	}
	else if (!CONNECTED_TO_LCD_BOARD)
	{
		MainActualState = MAIN_STATE_DISCONNECTED;
	}
	
	//run SPI process here for communication with LL16 
}



void StateControlledByUsb(void)
{
	if (CONNECTED_TO_LCD_BOARD)
	{
		if (!CONNECTED_TO_USB || GUI_TOLD_ME_TO_GO_STAND_ALONE)
		{
			//wake up LL16
			//add code here
			MainActualState = MAIN_STATE_CONTROLLED_BY_LCD_BOARD;
		}
	}
	else //!CONNECTED_TO_LCD_BOARD
	{
		if (!CONNECTED_TO_USB)
		{
			MainActualState = MAIN_STATE_DISCONNECTED;	
		}		
	}
		
	///*CDC_Engine();	//run CDC task
}









void SerialComm_PeriodicTask(void)
{	
 	if (MainActualState == MAIN_STATE_CONTROLLED_BY_USB)
	{
   				
		static uint_8 status = 0;///*
		//if there is data, read it
		if(g_recv_size)   //*
	  {
	    ///*InSize = CDC_GET_DATA(InBuffer);	
	    
	    /* Copy Received Buffer to InBuffer   
        for (status = 0; status < g_recv_size; status++)
        {
            InBuffer[status] = g_curr_recv_buf[status];
        }
         
       */  
        InSize = g_recv_size;
	    
	    
	    
		}
	}
	else if (MainActualState == MAIN_STATE_CONTROLLED_BY_LCD_BOARD)
	{
		//check if there is new data from the LCD board
		//add SPI receive function here	
	}


	//if there is data, parse it and execute command
	if (InSize>0)   ///*   g_recv_size
	{	
		//there is new data
		if (g_curr_recv_buf[PACKET_TYPE] == REQ)
		{
			if (ExecuteCommandReq[g_curr_recv_buf[COMMAND_OPCODE]] != NULL)
			{
				//check if OPCCODE is in a valid range
				if ((g_curr_recv_buf[COMMAND_OPCODE] <= LAST_COMMAND))
				{
					ExecuteCommandReq[g_curr_recv_buf[COMMAND_OPCODE]]();
				}
			}	
		}
		//else
		//{
			//packet type is not a request
		//}
		
		g_recv_size=0;
		InSize = 0;
	}
}

    





void SerialComm_SendData(void)
{
	  
	if (MainActualState == MAIN_STATE_CONTROLLED_BY_USB)
	{	
			
		if (OutSize>0)			 ///*  OutSize>0
		{			
			//TO DO:
			//check if previous data has been sent
			///*SEND_DATA_TO_HOST(OutBuffer, OutSize);
			
			USB_Class_CDC_Interface_DIC_Send_Data(CONTROLLER_ID,OutBuffer,OutSize);
      OutSize = 0;
		}
	}
	else if (MainActualState == MAIN_STATE_CONTROLLED_BY_LCD_BOARD)
	{
		//add SPI send function here	

	}	
}





#define MEDICAL_SENSOR_BOARD_IIC_ADDRESS		0x03

const UINT8 CmdEnableGlu	=	0x11;	
const UINT8 CmdEnableSpr	=	0x22;	
const UINT8 CmdEnablePox	=	0x33;	
const UINT8 CmdEnableEcg	=	0x44;	
const UINT8 CmdEnableBpm	=	0x55;	
const UINT8 CmdEnableHwt	=	0x66;	
const UINT8 CmdDisableAll	=	0xFF;



void EnableAFE(UINT8 measurementId)
{
	UINT8 delay=0xFF;
	
	AFE_DISABLE();		//power off board
	
	while(delay--)	//wait a few us
	{
		;
	}
	
	IIC_DISABLE();
	vfnIICV1Init();
	AFE_ENABLE();		//power on board
	
	switch (measurementId)		//send IIC data to select device
	{
		case NO_MEASUREMENT:
			break;
			
		case GLU_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, (UINT8*)&CmdEnableGlu, 1);
			break;
			
		case BPM_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, &CmdEnableBpm, 1);
			break;
						
		case ECG_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, &CmdEnableEcg, 1);
			break;
		
		case SPR_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, &CmdEnableSpr, 1);
			break;
			
		case HEIGHT_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, &CmdEnableHwt, 1);
			break;		
		
		case WEIGHT_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, &CmdEnableHwt, 1);
			break;		
		
		case TEMPERATURE_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, &CmdEnableHwt, 1);
			break;		
	}	
}


void SystemRestartReq(void)
{
	//stop all measurements
	GlucoseMeter_AbortMeasurement();

}

/**************************************************
		Glucose Meter
******************************************************/

/**
	Starts a glucose measurment
*/
void GluStartMeasurementReq(void)
{
	if (ActualMeasurement == NO_MEASUREMENT)
	{
		AFE_ENABLE();
		
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = GLU_START_MEASUREMENT;
		OutBuffer[OutSize++] = 1;	//data bytes

		if (GlucoseMeter_StartMeasurement() == TRUE)
		{
			OutBuffer[OutSize++] = ERROR_OK;
		}
		else
		{
			OutBuffer[OutSize++] = ERROR_BUSY;
		}

		(void)SerialComm_SendData();
		ActualMeasurement = GLU_MEASUREMENT;	
	}
}


/**
	Aborts a glucose measurement
*/
void GluAbortMeasurementReq(void)
{
	if (ActualMeasurement == GLU_MEASUREMENT)
	{		
		//execute command and send confirm
		GlucoseMeter_AbortMeasurement();
		
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = GLU_ABORT_MEASUREMENT;
		OutBuffer[OutSize++] = 0;	//data bytes
				
		(void)SerialComm_SendData();
		ActualMeasurement = NO_MEASUREMENT;
		
		AFE_DISABLE();
	}
}


/**
	Starts a calibrarion routine in the glucose meter
*/
void GluStartCalibrationReq(void)
{
	UINT16 knownGlucoseValue;

	if (ActualMeasurement == NO_MEASUREMENT)
	{
		AFE_ENABLE();
		
		knownGlucoseValue = InBuffer[DATA_PACKET]<<8;
		knownGlucoseValue += InBuffer[DATA_PACKET+1];

		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = GLU_START_CALIBRATION;
		OutBuffer[OutSize++] = 1;	//data bytes

		if (GlucoseMeter_StartCalibration(knownGlucoseValue) == TRUE)
		{
			OutBuffer[OutSize++] = ERROR_OK;
		}
		else
		{
			OutBuffer[OutSize++] = ERROR_BUSY;
		}
				
		(void)SerialComm_SendData();
		ActualMeasurement = GLU_MEASUREMENT;
	}
}

/**
	Sends an IND telling that the user has placed blood in the strip
*/
void GlucometerBloodDetectedInd(void)
{
	if (ActualMeasurement == GLU_MEASUREMENT)
	{		
		//Send GLU_BLOOD_DETECTED indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = GLU_BLOOD_DETECTED;
		OutBuffer[OutSize++] = 0;				//data bytes
		(void)SerialComm_SendData();
	}			
}


/**
	Sends an IND telling that the glucose test has finished.
	It sends to the host the glucose value measured.
*/
void GlucometerMeasurementOkInd(void)
{
	if (ActualMeasurement == GLU_MEASUREMENT)
	{		
		//Send  indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = GLU_MEASUREMENT_COMPLETE_OK;
		OutBuffer[OutSize++] = 2;	//data bytes
		OutBuffer[OutSize++] = GlucoseMeter_GlucoseValue >> 8;			//high byte
		OutBuffer[OutSize++] = GlucoseMeter_GlucoseValue & 0x00FF;		//low byte
		(void)SerialComm_SendData();
		ActualMeasurement = NO_MEASUREMENT;
		
		AFE_DISABLE();
	}
}


/**
	Sends an IND telling that the calibrarion routine finished correctly.
*/
void GlucometerCalibrationOkInd(void)
{
	if (ActualMeasurement == GLU_MEASUREMENT)
	{
		//Send  indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = GLU_CALIBRATION_COMPLETE_OK;
		OutBuffer[OutSize++] = 2;	//data bytes: calibration constant
		OutBuffer[OutSize++] = (UINT8)(GlucoseMeter_CalibrationConstant >> 8);		//high byte
		OutBuffer[OutSize++] = (UINT8)(GlucoseMeter_CalibrationConstant & 0x00FF);	//low byte
		(void)SerialComm_SendData();		
		ActualMeasurement = NO_MEASUREMENT;
		
		AFE_DISABLE();
	}
}

/**
	This function is only used during debug to send the actual ADC value read by the MCU.
	This function is activated if the GLU_DEBUG is defined in GlucoseMeter.h
*/
void GlucometerDebugNewDataReadyInd(void)
{
	if (ActualMeasurement == GLU_MEASUREMENT)
	{
		UINT8 i;
		
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = GLU_DEBUG_MODE_NEW_DATA_READY;
		OutBuffer[OutSize++] = GLU_GRAPH_DATA_ARRAY_SIZE;	//data bytes: real time graph
		
		
		for (i=0; i<GLU_GRAPH_DATA_ARRAY_SIZE; i++)
		{			
			OutBuffer[OutSize++] = GlucoseMeter_GraphDataToPc[i];
		}
		(void)SerialComm_SendData();		
	}
	
}






#endif



/*****************************************************************************
 * Local Functions
 *****************************************************************************/
/*****************************************************************************
 *
 *    @name     Init_Sys
 *
 *    @brief    This function Initializes the system
 *
 *    @param    None
 *
 *    @return   None
 *
 ****************************************************************************
 * Intializes the MCU, MCG, KBI, RTC modules
 ***************************************************************************/
static void Init_Sys(void)
{
    Mcu_Init(); /* initialize the MCU registers */
    MCG_Init(); /* initialize the MCG to generate 24MHz bus clock */
    //Kbi_Init(); /* initialize the KBI registers */
}

/*****************************************************************************
 * @name     MCU_Init
 *
 * @brief:   Initialization of the MCU.
 *
 * @param  : None
 *
 * @return : None
 *****************************************************************************
 * It will configure the MCU to disable STOP and COP Modules.
 * It also set the MCG configuration and bus clock frequency.
 ****************************************************************************/
static void Mcu_Init()
{


  /* Disable watchdog. */
    SOPT1 = 0x23; /* disable COP; enable bkgd, stop and wait mode */
 
    SOPT2 = 0x06; /* set right endianess for USB module */
    SOPT3 = SOPT3_SCI1PS_MASK;/* route TX1,RX1 to PTD6,PTD7 */

  /* Configure LED io pins to be outputs. 
   * EVB51JM128: LED to port mapping: 
   *  LED1  LED2 LED3 LED4 LED5 LED6 LED7 LED8
   *   |     |    |    |    |    |    |    |
   *   ^     ^    ^    ^    ^    ^    ^    ^
   *  PTE2  PTE3 PTF0 PTF1 PTF5 PTD2 PTC4 PTC6
   * DEMO51JM128: LED to port mapping: 
   *  LED1  LED2 LED3 LED4 LED5 LED6 LED7 LED8
   *   |     |    |    |    |    |    |    |
   *   ^     ^    ^    ^    ^    ^    ^    ^
   *  PTE2  PTE3 PTF0 PTF1 PTC2 PTC4 PTF5 PTD2
   */
  PTEDD= PTED_PTED2_MASK | PTED_PTED3_MASK;
  PTFDD= PTFD_PTFD0_MASK | PTFD_PTFD1_MASK | PTFD_PTFD5_MASK;
  PTCDD= PTCD_PTCD4_MASK 
 #ifdef EVB
          | PTCD_PTCD6_MASK
 #else
          | PTCD_PTCD2_MASK          
 #endif          
          ;
          
  PTDDD= PTDD_PTDD2_MASK;
  
  /* Enable internal pull-ups on port E pins to get switches working.
   * EVB51JM128: Button-to-port mapping:
   *  SW1   SW2   SW3   SW4
   *   |     |     |     |
   *   ^     ^     ^     ^
   *  PTG1  PTG2  PTG3  PTD5
   * DEMO51JM128: Button-to-port mapping:
   *  SW1   SW2   SW3   SW4
   *   |     |     |     |
   *   ^     ^     ^     ^
   *  PTG0  PTG1  PTG2  PTG3
   */
  PTGPE= PTGPE_PTGPE1_MASK | PTGPE_PTGPE2_MASK | PTGPE_PTGPE3_MASK
  #ifndef EVB
    | PTGPE_PTGPE0_MASK  
  #endif
   ;
  PTDPE= PTDPE_PTDPE5_MASK ;
  
  IRQSC = IRQSC_IRQIE_MASK | IRQSC_IRQPE_MASK; /*enable the IRQ interrupt for SW4 */    

}

/*****************************************************************************
 * @name     MCG_Init
 *
 * @brief:   Initialization of the Multiple Clock Generator.
 *
 * @param  : None
 *
 * @return : None
 *****************************************************************************
 * Provides clocking options for the device, including a phase-locked
 * loop(PLL) and frequency-locked loop (FLL) for multiplying slower reference
 * clock sources
 ****************************************************************************/
static void MCG_Init()
{

 /* Assume 12MHz external clock source connected. */
  
  /* In order to use the USB we need to enter PEE mode and MCGOUT set to 48 MHz. 
     Out of reset MCG is in FEI mode. */
  
  
  
  
  //------FBE MODE------  
  /* Assume 16MHz external clock source connected. */
  // RANGE = 1; HGO = 1; ERCLKEN=1; EREFS = 1; BDIV = 000
  
  MCGC2 = 0x36;
  
  
  
  // DIV32 =1
  MCGC3 = 0x10;
  
  // CLKS = 10; RDIV = 100; IREFS = 0;
  MCGC1 = 0xA0; 


  // Wait for Reference Status bit to update
  while (MCGSC_IREFST)
  {
  }
  // Wait for clock status bits to update 
  while (MCGSC_CLKST != 0b10) 
  {
  }

//------PBE MODE------ 

  // PLLS =1; DIV32 = 1; VDIV = 1001 
  MCGC3 = 0x5c;
  
  // wait for PLL status bit to update
  while (!MCGSC_PLLST) 
  {
  }
  // Wait for LOCK bit to set 
  while (!MCGSC_LOCK) 
  {
  }
  
//------PEE MODE------   

  // CLKS = 00; RDIV = 100; IREFS = 0
  MCGC1 = 0x20;

// Wait for clock status bits to update 
  while (MCGSC_CLKST != 0b11) 
  {
  }
}

